library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.std_logic_unsigned.all;

entity iec_c30_reg is
  port ( RESET             : in std_logic;
         H3_CLK            : in std_logic;
         GLO_CONT_SEL      : in std_logic;
         TIMER_CONT_SEL    : in std_logic;
         TIMER_PERIOD_SEL  : in std_logic;
         DATA_RX_SEL       : in std_logic;
         PH_RW             : in std_logic;
         PH_DATA_I         : in std_logic_vector(31 downto 0);
         B_FLAG            : in std_logic;
         LEFT_RIGHT        : in std_logic;
         DATA_W_EN         : in std_logic;
         DATA_IN           : in std_logic_vector(27 downto 0);

         IEC_RESET         : out std_logic;
         MODE              : out std_logic_vector(1 downto 0);
         CLKSRC            : out std_logic_vector(1 downto 0);
         INT_EN            : out std_logic;
         INT_CLK           : out std_logic;
         PH_DATA_O         : out std_logic_vector(31 downto 0));
end iec_c30_reg;

architecture beh of iec_c30_reg is

  signal glo_cont_val  : std_logic_vector(11 downto 0);
  signal timer_cont_val  : std_logic;
  signal timer_period_val  : std_logic_vector(3 downto 0);
  signal iec_rx_val  : std_logic_vector(27 downto 0);
  signal rx_val_empty  : std_logic;
  signal update_en  : std_logic;
  signal read_rx_data  : std_logic_vector(31 downto 0);

  signal timer_cnt  : std_logic_vector(3 downto 0);
  signal timer_cnt_clr  : std_logic;
  signal int_clk_tmp  : std_logic;
  signal read_len  : std_logic_vector(1 downto 0);
  signal sign_ext  : std_logic;
  signal rm  : std_logic;  -- read_rx_data msb

begin

  -- 11:RX_FULL,10:L_R,9:B_FLAG,8:INT_EN,7-6:CLKSRC,5:SIGN_EXT,4-3:READ_LEN,2-1:MODE,0:RESET
  glo_cont_reg: process (RESET, H3_CLK)
  begin
    if (RESET = '0') then
      glo_cont_val <= "000000000000";
    elsif (H3_CLK'event and H3_CLK = '1') then
      if (GLO_CONT_SEL = '1' and PH_RW = '0') then
        glo_cont_val(8 downto 0) <= PH_DATA_I(8 downto 0);
      end if;
      if (update_en = '1') then
        glo_cont_val(9) <= B_FLAG;
        glo_cont_val(10) <= LEFT_RIGHT;
      end if;
      if (DATA_RX_SEL = '1' and PH_RW = '1') then
        glo_cont_val(11) <= '0';  -- rx_val empty
      elsif (DATA_W_EN = '1' and rx_val_empty = '0') then
        glo_cont_val(11) <= '1';  -- rx_val full
      end if;
    end if;
  end process glo_cont_reg;

  timer_reg: process (RESET, H3_CLK)
  begin
    if (RESET = '0') then
      timer_cont_val <= '0';
      timer_period_val <= "0000";
    elsif (H3_CLK'event and H3_CLK = '1') then
      if (TIMER_CONT_SEL = '1' and PH_RW = '0') then
        timer_cont_val <= PH_DATA_I(0);
      end if;
      if (TIMER_PERIOD_SEL = '1' and PH_RW = '0') then
        timer_period_val <= PH_DATA_I(3 downto 0);
      end if;
    end if;
  end process timer_reg;

  timer_cnt_reg: process (RESET, H3_CLK)
  begin
    if (RESET = '0') then
      timer_cnt <= "0000";
    elsif (H3_CLK'event and H3_CLK = '1') then
      if (timer_cnt_clr = '1') then
        timer_cnt <= "0000";
      elsif (timer_cont_val = '1') then
        timer_cnt <= timer_cnt + '1';
      else
        timer_cnt <= "0000";
      end if;
    end if;
  end process timer_cnt_reg;

  timer_cnt_clr <= '1' when (timer_cont_val = '1' and (timer_cnt = timer_period_val)) else '0';

  int_clk_gen: process (RESET, H3_CLK)
  begin
    if (RESET = '0') then
      int_clk_tmp <= '0';
    elsif (H3_CLK'event and H3_CLK = '0') then
      if (timer_cont_val = '0') then
        int_clk_tmp <= '0';
      elsif (timer_cnt_clr = '1') then
        int_clk_tmp <= not int_clk_tmp;
      end if;
    end if;
  end process int_clk_gen;

  rx_val_empty_reg: process (RESET, H3_CLK)
  begin
    if (RESET = '0') then
      rx_val_empty <= '1';
    elsif (H3_CLK'event and H3_CLK = '1') then
      if (DATA_W_EN = '1') then                       -- write to rx_data reg.
        rx_val_empty <= '0';
      elsif (DATA_RX_SEL = '1' and PH_RW = '1') then
        rx_val_empty <= '1';                          -- read from rx_data reg.
      end if;
    end if;
  end process rx_val_empty_reg;

  update_en <= '1' when (DATA_W_EN = '1' and
                         (rx_val_empty = '1' or
                          (rx_val_empty = '0' and (DATA_RX_SEL = '1' and PH_RW = '1')))) else '0';

  iec_rx_reg: process (RESET, H3_CLK)
  begin
    if (RESET = '0') then
      iec_rx_val <= (others => '0');
    elsif (H3_CLK'event and H3_CLK = '1') then
      if (update_en = '1') then
        iec_rx_val <= DATA_IN;
      end if;
    end if;
  end process iec_rx_reg;

  rm <= iec_rx_val(23) when (sign_ext = '0') else '0';
  read_rx_data <= rm & rm & rm & rm & rm & rm & rm & rm & rm & rm & rm & rm & rm & rm & rm & rm &
                  iec_rx_val(23 downto 8) when (read_len = "00") else   -- 16 bit
                  rm & rm & rm & rm & rm & rm & rm & rm & rm & rm & rm & rm &
                  iec_rx_val(23 downto 4) when (read_len = "01") else   -- 20 bit
                  rm & rm & rm & rm & rm & rm & rm & rm &
                  iec_rx_val(23 downto 0) when (read_len = "10") else   -- 24 bit
                  "0000" & iec_rx_val;                                  -- 28 bit

  ph_data_out: process (GLO_CONT_SEL, TIMER_CONT_SEL, TIMER_PERIOD_SEL, DATA_RX_SEL, PH_RW)
  begin
    if (GLO_CONT_SEL = '1' and PH_RW = '1') then
      PH_DATA_O <= "00000000000000000000" & glo_cont_val;
    elsif (TIMER_CONT_SEL = '1' and PH_RW = '1') then
      PH_DATA_O <= "0000000000000000000000000000000" & timer_cont_val;
    elsif (TIMER_PERIOD_SEL = '1' and PH_RW = '1') then
      PH_DATA_O <= "0000000000000000000000000000" & timer_period_val;
    elsif (DATA_RX_SEL = '1' and PH_RW = '1') then
      PH_DATA_O <= read_rx_data;
    else
      PH_DATA_O <= (others => 'Z');
    end if;
  end process ph_data_out;

  IEC_RESET <= glo_cont_val(0);           -- low active
  MODE <= glo_cont_val(2 downto 1);
  read_len <= glo_cont_val(4 downto 3);
  sign_ext <= glo_cont_val(5);            -- low active
  CLKSRC <= glo_cont_val(7 downto 6);
  INT_EN <= glo_cont_val(8);
  INT_CLK <= int_clk_tmp;

end beh;
