//
// Copyright (c) 1997-2001 UCR. Permission to copy is granted
// provided that this header remains intact. This software is
// provided with no warranties.
//

//----------------------------------------------------------------------------

#include <cstdlib>
#include "tech.h"
#include "bussim.h"

//----------------------------------------------------------------------------

BusSimulator::BusSimulator() {
    
    Reset(32, Binary, 32, Binary, 3.3);
}

//----------------------------------------------------------------------------

BusSimulator::~BusSimulator() {
}

//----------------------------------------------------------------------------

void BusSimulator::Reset(unsigned aw, Encoding ae, 
                         unsigned dw, Encoding de,
                         double _voltage) {
    
    addrBitToggles = 0;
    dataBitToggles = 0;
    
    addrState.width = aw;
    addrState.encoding = ae;
    addrState.packets = 32 / addrState.width;
    addrState.mask = 0xffffffffu >> (32 - addrState.width);
    addrState.state = 0;
    addrState.invert = false;
    
    dataState.width = dw;
    dataState.encoding = de;
    dataState.packets = 32 / dataState.width;
    dataState.mask = 0xffffffffu >> (32 - dataState.width);
    dataState.state = 0;
    dataState.invert = false;
    
    voltage = _voltage;
}

//----------------------------------------------------------------------------

double BusSimulator::ComputePower() {
    
    double p;
    
    // average wire capacitance
    p = (dataState.width + addrState.width) * AvgBusWireLen;
    p = (p / Tech) * CapWire;
    
    // compute total power
    p = p * voltage * voltage * .5;
    
    // adjust for traffic
    return p * (addrBitToggles + dataBitToggles);
}

//----------------------------------------------------------------------------

void BusSimulator::Transfer(unsigned long a) {
    
    unsigned long i;
    unsigned long temp;
    unsigned long d = rand(); // random data assumption
    
    for(i=0; i<addrState.packets; i++) {
        
        temp = (a >> (addrState.width * i)) & addrState.mask;
        switch( addrState.encoding ) {
            
        case Binary: 
            addrBitToggles += TransferBinary(temp, addrState); 
            break;
            
        case Gray:   
            addrBitToggles += TransferGray(temp, addrState); 
            break;
            
        case Invert: 
            addrBitToggles += TransferInvert(temp, addrState); 
            break;
        }
    }
    
    for(i=0; i<dataState.packets; i++) {
        
        temp = (a >> (dataState.width * i)) & dataState.mask;
        switch( dataState.encoding ) {
            
        case Binary: 
            dataBitToggles += TransferBinary(temp, dataState); 
            break;
            
        case Gray:   
            dataBitToggles += TransferGray(temp, dataState); 
            break;
            
        case Invert: 
            dataBitToggles += TransferInvert(temp, dataState); 
            break;
        }
    }
}

//----------------------------------------------------------------------------

unsigned long BusSimulator::TransferBinary(unsigned long v, BusState &bs) {
    
    unsigned long toggles;
    
    toggles = HamDist(bs.state, v, bs.width);
    bs.state = v;
    return toggles;
}

//----------------------------------------------------------------------------

unsigned long BusSimulator::TransferGray(unsigned long v, BusState &bs) {
    
    unsigned long toggles;
    
    toggles = abs(bs.state - v) % (bs.width / 2);
    bs.state = v;
    return toggles;
}

//----------------------------------------------------------------------------

unsigned long BusSimulator::TransferInvert(unsigned long v, BusState &bs) {
    
    unsigned long toggles;
    unsigned long hd;
    
    hd = HamDist(bs.state, v, bs.width);
    if( hd > bs.width/2 ) {
        
        toggles = (bs.width - hd) + (bs.invert == false ? 1 : 0);
        bs.state = !v & bs.mask;
        bs.invert = true;
    }
    else {
        
        toggles = hd + (bs.invert == true ? 1 : 0);
        bs.state = v;
        bs.invert = false;
    }
    return toggles;
}

//----------------------------------------------------------------------------

unsigned long BusSimulator::HamDist(unsigned long v1, 
                                    unsigned long v2, 
                                    unsigned width) {
    
    unsigned long hd = 0;
    
    for(unsigned long i=0; i<width; i++) {
        
        if( ((v1 >> i) & 1) != ((v2 >> i) & 1) ) hd++;
    }
    return hd;
}