//
// Copyright (c) 1997-2001 Tony Givargis. Permission to copy is
// granted provided that this header remains intact. This software
// is provided with no warranties.
//

//----------------------------------------------------------------------------

#include <cstring>
#include <cmath>
#include "tech.h"
#include "cachesim.h"

//----------------------------------------------------------------------------

CacheSimulator::CacheSimulator() {
    
    entries = 0;
    Reset(16384, 16, 1, 3.3);
};

//----------------------------------------------------------------------------

CacheSimulator::~CacheSimulator() {
    
    for(unsigned i=0; i<szEntries; i++) {
        
        delete[] entries[i].cnt;
        delete[] entries[i].tag;
        delete[] entries[i].valid;
        delete[] entries[i].dirty;
    }
    delete[] entries;
}

//----------------------------------------------------------------------------

void CacheSimulator::Reset(unsigned _size, 
                           unsigned _line, 
                           unsigned _assoc,
                           double _voltage) {
    
    size = _size;
    line = _line;
    assoc = _assoc;
    voltage = _voltage;
    
    numRead = 0;
    numWrite = 0;
    numReadMiss = 0;
    numWriteMiss = 0;
    numWriteBack = 0;
    
    // deallocate
    if( entries ) {
        
        for(unsigned i=0; i<szEntries; i++) {
            
            delete[] entries[i].cnt;
            delete[] entries[i].tag;
            delete[] entries[i].valid;
            delete[] entries[i].dirty;
        }
        delete[] entries;
    }
    
    // allocate entries
    szEntries = size / (line * assoc);
    entries = new Entry[szEntries];
    
    // allocate lines within entries
    for(unsigned i=0; i<szEntries; i++) {
        
        entries[i].cnt = new unsigned[assoc];
        entries[i].tag = new unsigned long[assoc];
        entries[i].valid = new bool[assoc];
        entries[i].dirty = new bool[assoc];
        
        memset(entries[i].cnt, 0, sizeof(unsigned) * assoc);
        memset(entries[i].tag, 0, sizeof(unsigned long) * assoc);
        memset(entries[i].valid, 0, sizeof(bool) * assoc);
        memset(entries[i].dirty, 0, sizeof(bool) * assoc);
    }
}

//----------------------------------------------------------------------------

double CacheSimulator::ComputePower() {
    
    double p1, p2, p3, p;
    double offset;
    double index;
    double tag;
    double colWidth, rowWidth;
    
    offset = Log2(line);
    index = Log2(size / (line * assoc));
    tag = 32 - index - offset;
    
    colWidth = 2 * (1 + 1 + tag + (8 * line)) * assoc;
    rowWidth = size / (line * assoc);
    
    // cache storage
    p1 = 
        (0.5 * colWidth * CapTran) + 
        (colWidth * Tech * CapWire) +
        ((0.5 * colWidth) * rowWidth * Tech * CapWire);
    
    // tag comparators
    p2 = (assoc * tag * 4) * CapTran;
    
    // row select
    p3 = (0.5 * pow(2, index)) * CapTran;
    
    // total
    p = ((p1 + p2 + p3) * voltage * voltage * 0.5);
    
    // adjust for traffic
    return p * (numRead + numWrite + numReadMiss + numWriteMiss);
}

//----------------------------------------------------------------------------

int CacheSimulator::Read(unsigned long a) {
    
    numRead++;
    return ReadWrite(a, false);
}

//----------------------------------------------------------------------------

int CacheSimulator::Write(unsigned long a) {
    
    numWrite++;
    return ReadWrite(a, true);
}

//----------------------------------------------------------------------------

int CacheSimulator::ReadWrite(unsigned long a, bool dirty) {
    
    unsigned long idx = (a / line) % szEntries;
    unsigned long tag = (a / line) / szEntries;
    unsigned i;
    
    // in cache and valid
    for(i=0; i<assoc; i++) {
        
        if( entries[idx].tag[i] == tag && entries[idx].valid[i] ) {
            
            entries[idx].cnt[i] = 1 + GetMaxCnt(idx);
            entries[idx].dirty[i] |= dirty;
            return 0;
        }
    }
    
    // update statistics
    if( dirty ) {
        
        numWriteMiss++;
    }
    else {
        
        numReadMiss++;
    }
    
    // not in cache, space available
    for(i=0; i<assoc; i++) {
        
        if( !entries[idx].valid[i] ) {
            
            entries[idx].cnt[i] = 1 + GetMaxCnt(idx);
            entries[idx].tag[i] = tag;
            entries[idx].valid[i] = true;
            entries[idx].dirty[i] = dirty;
            return 1;
        }
    }
    
    // not in cache, no space available
    i = GetLru(idx);
    if( !entries[idx].dirty[i] ) {
        
        entries[idx].cnt[i] = 1 + GetMaxCnt(idx);
        entries[idx].tag[i] = tag;
        entries[idx].dirty[i] = dirty;
        return 1;
    }
    else {
        
        numWriteBack++;
        entries[idx].cnt[i] = 1 + GetMaxCnt(idx);
        entries[idx].tag[i] = tag;
        entries[idx].dirty[i] = dirty;
        return 2;
    }
}

//----------------------------------------------------------------------------

unsigned CacheSimulator::GetLru(unsigned idx) {
    
    unsigned i, temp, lru = 0;
    
    // find lru
    for(i=0; i<assoc; i++) {
        
        if( entries[idx].cnt[lru] > entries[idx].cnt[i] ) {
            
            lru = i;
        }
    }
    
    // bump down
    temp = entries[idx].cnt[lru];
    for(i=0; i<assoc; i++) {
        
        entries[idx].cnt[i] -= temp;
    }
    return lru;
}

//----------------------------------------------------------------------------

unsigned CacheSimulator::GetMaxCnt(unsigned idx) {
    
    unsigned i, maxCnt = 0;
    
    for(i=0; i<assoc; i++) {
        
        if( entries[idx].cnt[i] > maxCnt ) {
            
            maxCnt = entries[idx].cnt[i];
        }
    }
    return maxCnt;
}

//----------------------------------------------------------------------------

double CacheSimulator::Log2(double x) {
    
    static const double LogOf2 = log(2.0);
    
    return log(x) / LogOf2;
}