//
// Copyright (c) 1997-2001 Tony Givargis. Permission to copy is
// granted provided that this header remains intact. This software
// is provided with no warranties.
//

//----------------------------------------------------------------------------

#include <cassert>
#include "explorer.h"

//----------------------------------------------------------------------------

#define FOR_ALL_POINTS \
for(idx[ 0]=0; idx[ 0]<lmt[ 0]; idx[ 0]++) \
for(idx[ 1]=0; idx[ 1]<lmt[ 1]; idx[ 1]++) \
for(idx[ 2]=0; idx[ 2]<lmt[ 2]; idx[ 2]++) \
for(idx[ 3]=0; idx[ 3]<lmt[ 3]; idx[ 3]++) \
for(idx[ 4]=0; idx[ 4]<lmt[ 4]; idx[ 4]++) \
for(idx[ 5]=0; idx[ 5]<lmt[ 5]; idx[ 5]++) \
for(idx[ 6]=0; idx[ 6]<lmt[ 6]; idx[ 6]++) \
for(idx[ 7]=0; idx[ 7]<lmt[ 7]; idx[ 7]++) \
for(idx[ 8]=0; idx[ 8]<lmt[ 8]; idx[ 8]++) \
for(idx[ 9]=0; idx[ 9]<lmt[ 9]; idx[ 9]++) \
for(idx[10]=0; idx[10]<lmt[10]; idx[10]++) \
for(idx[11]=0; idx[11]<lmt[11]; idx[11]++) \
for(idx[12]=0; idx[12]<lmt[12]; idx[12]++) \
for(idx[13]=0; idx[13]<lmt[13]; idx[13]++) \
for(idx[14]=0; idx[14]<lmt[14]; idx[14]++) \
for(idx[15]=0; idx[15]<lmt[15]; idx[15]++) \
for(idx[16]=0; idx[16]<lmt[16]; idx[16]++) \
for(idx[17]=0; idx[17]<lmt[17]; idx[17]++) \
for(idx[18]=0; idx[18]<lmt[18]; idx[18]++)

//----------------------------------------------------------------------------

bool operator < (const Point &p1, const Point &p2) {
    
    return p1.time < p2.time;
}

//----------------------------------------------------------------------------

Cluster::Cluster() {
}

//----------------------------------------------------------------------------

Cluster::~Cluster() {
}

//----------------------------------------------------------------------------

void Cluster::Clear() {
    
    points.clear();
    for(int i=0; i<TotalParameters; i++) {
        
        memberSet[i] = false;
    }
}

//----------------------------------------------------------------------------

void Cluster::AddParameter(int p) {
    
    memberSet[p] =  true;
}

//----------------------------------------------------------------------------

void Cluster::UnionWith(const Cluster &c) {
    
    for(int i=0; i<TotalParameters; i++) {
        
        memberSet[i] = memberSet[i] || c.memberSet[i];
    }
}

//----------------------------------------------------------------------------

bool Cluster::IsMember(int p) const {
    
    return memberSet[p] ? true : false;
}

//----------------------------------------------------------------------------

bool Cluster::IsDisjoint(const Cluster &c) const {
    
    for(int i=0; i<TotalParameters; i++) {
        
        if( c.memberSet[i] && memberSet[i] ) {
            
            return false;
        }
    }
    return true;
}

//----------------------------------------------------------------------------

Explorer::Explorer(Loader *_loader) {
    
    loader = _loader;
    
    assert( loader );
    
    loader->totalPoints = 0;
    loader->currentPoint = 0;
}

//----------------------------------------------------------------------------

Explorer::~Explorer() {
}

//----------------------------------------------------------------------------

void Explorer::Explore() {
    
    Cluster c1, c2;
    
    loader->Reset();
    loader->totalPoints = 0;
    loader->currentPoint = 0;
    
    ComputeParameterSpace();
    ComputeInitialClusters();
    
    // phase 1
    for(list<Cluster>::iterator i=clusters.begin(); i!=clusters.end(); ++i) {
        
        ExploreClusterInitial(*i);
        SortParetoOptimal(*i);
    }
    
    // phase 2
    while( clusters.size() != 1 ) {
        
		assert( clusters.begin() != clusters.end() );
        
		c1 = *clusters.begin(); 
		clusters.pop_front();
		
		assert( clusters.begin() != clusters.end() );
        
		c2 = *clusters.begin(); 
		clusters.pop_front();
        
        assert( c1.IsDisjoint(c2) );
        
        ExploreClusterSubsequent(c1, c2);
        c1.UnionWith(c2);
        SortParetoOptimal(c1);
        clusters.push_back(c1);
    }
    
    // generate report
    c1 = *clusters.begin();
    for(list<Point>::iterator j=c1.points.begin(); j!=c1.points.end(); ++j) {
        
        loader->Pareto((*j).param, (*j).power, (*j).time);
    }
    clusters.clear();
}

//----------------------------------------------------------------------------

void Explorer::ComputeParameterSpace() {
    
    memset(val, 0, sizeof(val));
    memset(lmt, 0, sizeof(lmt));
    
    for(int i=0; i<TotalParameters; i++) {
        
        for(int j=0; j<PointsPerParameter[i]; j++) {
            
            if( !loader->psData->space[i][j] ) continue;
            
            switch( i ) {
                
            case  0: val[i][lmt[i]++] = (128 << j); break;
            case  1: val[i][lmt[i]++] = (  4 << j); break;
            case  2: val[i][lmt[i]++] = (  1 << j); break;
            case  3: val[i][lmt[i]++] = (128 << j); break;
            case  4: val[i][lmt[i]++] = (  4 << j); break;
            case  5: val[i][lmt[i]++] = (  1 << j); break;
            case  6: val[i][lmt[i]++] = (  4 << j); break;
            case  7: val[i][lmt[i]++] = (       j); break;
            case  8: val[i][lmt[i]++] = (  4 << j); break;
            case  9: val[i][lmt[i]++] = (       j); break;
            case 10: val[i][lmt[i]++] = (  4 << j); break;
            case 11: val[i][lmt[i]++] = (       j); break;
            case 12: val[i][lmt[i]++] = (  4 << j); break;
            case 13: val[i][lmt[i]++] = (       j); break;
            case 14: val[i][lmt[i]++] = (  4 << j); break;
            case 15: val[i][lmt[i]++] = (       j); break;
            case 16: val[i][lmt[i]++] = (  4 << j); break;
            case 17: val[i][lmt[i]++] = (       j); break;
            case 18: val[i][lmt[i]++] = (10  +  j); break;
            default: assert( false );
            }
        }
    }
}

//----------------------------------------------------------------------------

void Explorer::ComputeInitialClusters() {
    
    int i;
    Cluster c;
    
    for(i=0; i<2; i++) {
        
        c.Clear();
        c.AddParameter(i * 3 + 0);
        c.AddParameter(i * 3 + 1);
        c.AddParameter(i * 3 + 2);
        clusters.push_back(c);
    }
    
    for(i=0; i<6; i++) {
        
        c.Clear();
        c.AddParameter(i * 2 + 6);
        c.AddParameter(i * 2 + 7);
        clusters.push_back(c);
    }
    
    c.Clear();
    c.AddParameter(18);
    clusters.push_back(c);
}

//----------------------------------------------------------------------------

void Explorer::ExploreClusterInitial(Cluster &c) {
    
    Point p;
    unsigned long totalPoints, i;
    int idx[TotalParameters];
    int copyOfLmt[TotalParameters];
    
    memcpy(copyOfLmt, lmt, sizeof(copyOfLmt));
    
    totalPoints = 1;
    for(i=0; i<TotalParameters; i++) {
        
        if( !c.IsMember(i) ) {
            
            lmt[i] = 1;
            continue;
        }
        totalPoints *= lmt[i];
    }
    loader->totalPoints += totalPoints;
    
    FOR_ALL_POINTS {
        
        loader->currentPoint++;
        
        for(i=0; i<TotalParameters; i++) {
            
            p.param[i] = val[i][idx[i]];
        }
        
        if( loader->mvm->Reset(
            static_cast<int>      (p.param[ 0]),
            static_cast<int>      (p.param[ 1]),
            static_cast<int>      (p.param[ 2]),
            static_cast<int>      (p.param[ 3]),
            static_cast<int>      (p.param[ 4]),
            static_cast<int>      (p.param[ 5]),
            static_cast<int>      (p.param[ 6]),
            static_cast<Encoding> (p.param[ 7]),
            static_cast<int>      (p.param[ 8]),
            static_cast<Encoding> (p.param[ 9]),
            static_cast<int>      (p.param[10]),
            static_cast<Encoding> (p.param[11]),
            static_cast<int>      (p.param[12]),
            static_cast<Encoding> (p.param[13]),
            static_cast<int>      (p.param[14]),
            static_cast<Encoding> (p.param[15]),
            static_cast<int>      (p.param[16]),
            static_cast<Encoding> (p.param[17]),
            static_cast<int>      (p.param[18]) / 10.0) ) {
            
            loader->mvm->Execute();
            loader->Report();
            
            p.power = loader->mvm->power;
            p.time = loader->mvm->exeTime;
            c.points.push_back(p);
        }
    }
    
    memcpy(lmt, copyOfLmt, sizeof(copyOfLmt));
}

//----------------------------------------------------------------------------

void Explorer::ExploreClusterSubsequent(Cluster &c1, Cluster &c2) {
    
    Point p;
    list<Point> points;
    list<Point>::iterator i, j;
    
    loader->totalPoints += c1.points.size() * c2.points.size();
    
    for(i=c1.points.begin(); i!=c1.points.end(); ++i) {
        
        for(j=c2.points.begin(); j!=c2.points.end(); ++j) {
            
            for(unsigned k=0; k<TotalParameters; k++) {
                
                if( c1.IsMember(k) ) {
                    
                    p.param[k] = (*i).param[k];
                }
                else if( c2.IsMember(k) ) {
                    
                    p.param[k] = (*j).param[k];
                }
                else {
                    
                    p.param[k] = val[k][0];
                }
            }
            
            loader->currentPoint++;
            
            if( loader->mvm->Reset(
                static_cast<int>      (p.param[ 0]),
                static_cast<int>      (p.param[ 1]),
                static_cast<int>      (p.param[ 2]),
                static_cast<int>      (p.param[ 3]),
                static_cast<int>      (p.param[ 4]),
                static_cast<int>      (p.param[ 5]),
                static_cast<int>      (p.param[ 6]),
                static_cast<Encoding> (p.param[ 7]),
                static_cast<int>      (p.param[ 8]),
                static_cast<Encoding> (p.param[ 9]),
                static_cast<int>      (p.param[10]),
                static_cast<Encoding> (p.param[11]),
                static_cast<int>      (p.param[12]),
                static_cast<Encoding> (p.param[13]),
                static_cast<int>      (p.param[14]),
                static_cast<Encoding> (p.param[15]),
                static_cast<int>      (p.param[16]),
                static_cast<Encoding> (p.param[17]),
                static_cast<int>      (p.param[18]) / 10.0) ) {
                
                loader->mvm->Execute();
                loader->Report();
                
                p.power = loader->mvm->power;
                p.time = loader->mvm->exeTime;
                points.push_back(p);
            }           
        }
    }
    c1.points = points;
}

//----------------------------------------------------------------------------

void Explorer::SortParetoOptimal(Cluster &c) {
    
    double minPower = 1e100;
    list<Point>::iterator i;
    
    c.points.sort();
    
    for(i=c.points.begin(); i!=c.points.end(); ) {
        
        if( (*i).power < minPower ) {
            
            minPower = (*i++).power;
        }
        else {
            
            i = c.points.erase(i);
        }
    }
    
    assert( c.points.size() );
}