//
// Copyright (c) 1997-2001 Tony Givargis. Permission to copy is
// granted provided that this header remains intact. This software
// is provided with no warranties.
//

//----------------------------------------------------------------------------

#include "app.h"
#include "child.h"
#include "psdlg.h"
#include "resource.h"
#include "frame.h"

//----------------------------------------------------------------------------

static const unsigned Indicators[] = {
    
    ID_SEPARATOR, ID_SEPARATOR, ID_SEPARATOR, ID_SEPARATOR
};

//----------------------------------------------------------------------------

SimulateThread::SimulateThread(Loader *loader) {
    
    simulator = new Simulator(loader);
}

//----------------------------------------------------------------------------

SimulateThread::~SimulateThread() {
    
    delete simulator;
}

//----------------------------------------------------------------------------

void SimulateThread::Run() {
    
    simulator->Simulate();
}

//----------------------------------------------------------------------------

ExploreThread::ExploreThread(Loader *loader) {
    
    explorer = new Explorer(loader);
}

//----------------------------------------------------------------------------

ExploreThread::~ExploreThread() {
    
    delete explorer;
}

//----------------------------------------------------------------------------

void ExploreThread::Run() {
    
    explorer->Explore();
}

//----------------------------------------------------------------------------

Frame::Frame() : CMDIFrameWnd() {
    
    char currDir[256];
    
    GetCurrentDirectory(sizeof(currDir), currDir);  
    loader = new Loader(currDir);
    simThread = new SimulateThread(loader);
    expThread = new ExploreThread(loader);
    applicationLoaded = false;
    state = 0;
}

//----------------------------------------------------------------------------

Frame::~Frame() {
    
    delete loader;
    delete simThread;
    delete expThread;
}

//----------------------------------------------------------------------------

int Frame::OnCreate(LPCREATESTRUCT lpCreateStruct) {
    
    DWORD w = 0;
    
    w |= TBSTYLE_FLAT, WS_CHILD | WS_VISIBLE | CBRS_TOP;
    w |= CBRS_GRIPPER | CBRS_SIZE_DYNAMIC;
    
    if( CMDIFrameWnd::OnCreate(lpCreateStruct) != -1 ) {
        
        statusBar.Create(this);
        statusBar.SetIndicators(Indicators, 4);
        statusBar.SetPaneInfo(1, ID_SEPARATOR, SBPS_NORMAL, 100);
        statusBar.SetPaneInfo(2, ID_SEPARATOR, SBPS_NORMAL, 100);
        statusBar.SetPaneInfo(3, ID_SEPARATOR, SBPS_NORMAL, 250);
        return 0;
    }
    return -1;
}

//----------------------------------------------------------------------------

void Frame::OnClose() {
    
    if( simThread->IsRunning() ) {
        
        MessageBox("Can't exit while simulating!");
    }
    else if( expThread->IsRunning() ) {
        
        MessageBox("Can't quit while exploring!");
    }
    else {
        
        CMDIFrameWnd::OnClose();
    }
}

//----------------------------------------------------------------------------

void Frame::OnTimer(UINT id) {
    
    char buf[256];
    
    if( simThread->IsRunning() ) {
        
        sprintf(buf, "Simulating %d sec", ++runTime);
        statusBar.SetPaneText(1, buf);
        sprintf(buf, "%lu/%lu", loader->currentPoint, loader->totalPoints);
        statusBar.SetPaneText(2, buf);
    }
    else if( expThread->IsRunning() ) {
        
        sprintf(buf, "Exploring %d sec", ++runTime);
        statusBar.SetPaneText(1, buf);
        sprintf(buf, "%lu/%lu", loader->currentPoint, loader->totalPoints);
        statusBar.SetPaneText(2, buf);
    }
    else {
        
        if( state & 3 ) new ChildWnd(this, loader->outFileName);
        if( state & 3 ) new ChildWnd(this, loader->repFileName);
        if( state & 2 ) new ChildWnd(this, loader->parFileName);
        
        sprintf(buf, "Done! %d sec", runTime);
        statusBar.SetPaneText(1, buf);
        sprintf(buf, "%lu/%lu", loader->currentPoint, loader->totalPoints);
        statusBar.SetPaneText(2, buf);
        KillTimer(id);
        state = 0;
    }
}

//----------------------------------------------------------------------------

void Frame::OnLoad() {
    
    char buf[256];
    CFileDialog fd(true, ".c", 0, 0, "C Source Files (*.c)|*.c||", this);
    
    if( fd.DoModal() == IDOK ) {
        
        if( loader->Load(fd.GetFileName(), buf) ) {
            
            sprintf(buf, "code/data=%lu/%lu (byte) - %s", 
                loader->ma->codeAddress * 4,
                loader->ma->dataAddress,
                fd.GetFileName());
            statusBar.SetPaneText(3, buf);
            applicationLoaded = true;
        }
        else {
            
            applicationLoaded = false;
            MessageBox(buf);
            new ChildWnd(this, loader->errFileName);
        }
    }
}

//----------------------------------------------------------------------------

void Frame::OnLoadUpdate(CCmdUI *pCmdUI) {
    
    bool sr, er;
    
    sr = simThread->IsRunning();
    er = expThread->IsRunning();
    pCmdUI->Enable( !sr && !er );
}

//----------------------------------------------------------------------------

void Frame::OnSetParameter() {
    
    ParameterSpaceDialog(this, loader->psData).DoModal();
}

//----------------------------------------------------------------------------

void Frame::OnSetParameterUpdate(CCmdUI *pCmdUI) {
    
    bool sr, er;
    
    sr = simThread->IsRunning();
    er = expThread->IsRunning();
    pCmdUI->Enable( !sr && !er );
}

//----------------------------------------------------------------------------

void Frame::OnAbort() {
    
    simThread->Stop();
    expThread->Stop();
    KillTimer(1000);
    statusBar.SetPaneText(1, "Aborted!");
}

//----------------------------------------------------------------------------

void Frame::OnAbortUpdate(CCmdUI *pCmdUI) {
    
    bool sr, er;
    
    sr = simThread->IsRunning();
    er = expThread->IsRunning();
    pCmdUI->Enable( sr || er );
}

//----------------------------------------------------------------------------

void Frame::OnSimulate() {
    
    runTime = 0;
    simThread->Start();
    SetTimer(1000, 1000, 0);
    state |= 1;
}

//----------------------------------------------------------------------------

void Frame::OnSimulateUpdate(CCmdUI *pCmdUI) {
    
    bool sr, er;
    
    sr = simThread->IsRunning();
    er = expThread->IsRunning();
    pCmdUI->Enable( !sr && !er && applicationLoaded );
}

//----------------------------------------------------------------------------

void Frame::OnExplore() {
    
    runTime = 0;
    expThread->Start();
    SetTimer(1000, 1000, 0);
    state |= 2;
}

//----------------------------------------------------------------------------

void Frame::OnExploreUpdate(CCmdUI *pCmdUI) {
    
    bool sr, er;
    
    sr = simThread->IsRunning();
    er = expThread->IsRunning();
    pCmdUI->Enable( !sr && !er && applicationLoaded );
}

//----------------------------------------------------------------------------

void Frame::OnAbout() {
    
    CDialog(IDD_DIALOG1, this).DoModal();
}

//-----------------------------------------------------------------------------

BEGIN_MESSAGE_MAP(Frame, CMDIFrameWnd)
ON_WM_CREATE()
ON_WM_CLOSE()
ON_WM_TIMER()
ON_COMMAND(ID_COMMAND_LOAD, OnLoad)
ON_UPDATE_COMMAND_UI(ID_COMMAND_LOAD, OnLoadUpdate)
ON_COMMAND(ID_COMMAND_SETPARAMETER, OnSetParameter)
ON_UPDATE_COMMAND_UI(ID_COMMAND_SETPARAMETER, OnSetParameterUpdate)
ON_COMMAND(ID_COMMAND_ABORT, OnAbort)
ON_UPDATE_COMMAND_UI(ID_COMMAND_ABORT, OnAbortUpdate)
ON_COMMAND(ID_COMMAND_SIMULATE, OnSimulate)
ON_UPDATE_COMMAND_UI(ID_COMMAND_SIMULATE, OnSimulateUpdate)
ON_COMMAND(ID_COMMAND_EXPLORE, OnExplore)
ON_UPDATE_COMMAND_UI(ID_COMMAND_EXPLORE, OnExploreUpdate)
ON_COMMAND(ID_HELP_ABOUT, OnAbout)
END_MESSAGE_MAP()
IMPLEMENT_DYNAMIC(Frame, CMDIFrameWnd)